package gov.va.vamf.scheduling.facility.datalayer;

import com.agilex.healthcare.utility.NullChecker;
import gov.va.vamf.scheduling.facility.domain.CDWFacilities;
import gov.va.vamf.scheduling.facility.domain.CDWFacility;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

@Component
public class FacilityCache {

	private Map<Integer, CDWFacilities> facilityObjMap;
	private CdwFacilityDataLayer facilityDataLayer;

	@Autowired
	private FacilityCache(CdwFacilityDataLayer facilityDataLayer) {
		this.facilityDataLayer = facilityDataLayer;
		this.updateFacilityCache();
	}

	/**
	 * We update the cache initially on deploy and then every day at 08:00:00 UTC time.
	 */
	@Scheduled(cron = "0 0 8 * * *")
	public void updateFacilityCache() {
		facilityObjMap = new ConcurrentHashMap<>();
		CDWFacilities facilities = this.facilityDataLayer.fetchFacilities();
		addFacilitiesToCache(facilities);

		long now = System.currentTimeMillis() / 1000;
		System.out.println("Updated facility cache with refresh from CDW - UTC DateTime " + now);
	}

	/**
	 * This method adds facilities to the cache grouping by parentSiteCode in which the parentSiteCode is the key and the
	 * values are the parent and its children.
	 * @param facilities
	 */
	public void addFacilitiesToCache(CDWFacilities facilities) {
		if (facilities != null) {
			for (CDWFacility facility : facilities) {
				CDWFacilities facilityWithChildren = this.facilityObjMap.get(facility.getParentSiteCode().hashCode());

				if (NullChecker.isNotNullish(facilityWithChildren)) {
					facilityWithChildren.add(facility);

				} else {
					facilityWithChildren = new CDWFacilities();
					facilityWithChildren.add(facility);
				}

				this.facilityObjMap.put(facility.getParentSiteCode().hashCode(), facilityWithChildren);
			}
		}
	}
	
	public CDWFacilities getFacilities(String siteCode) {
		if (siteCode == null)
			return null;
		return facilityObjMap.get(siteCode.hashCode());
	}
}
